//
//  PredictionHistoryManager.swift
//  death_app Watch App
//
//  Created by Task-Master AI
//

import Foundation
import CoreData
import SwiftUI

class PredictionHistoryManager: ObservableObject {
    static let shared = PredictionHistoryManager()
    private let databaseService = DatabaseService.shared
    
    @Published var historicalData: [PredictionHistoryEntry] = []
    @Published var trendAnalysis: TrendAnalysis?
    @Published var isLoading = false
    
    private init() {}
    
    // MARK: - Public Methods
    
    func fetchHistoricalData(timeframe: HistoricalTimeframe) async {
        await MainActor.run {
            isLoading = true
        }
        
        do {
            let predictions = try await databaseService.fetchPredictionHistory(timeframe: timeframe)
            let analysis = calculateTrendAnalysis(from: predictions)
            
            await MainActor.run {
                self.historicalData = predictions
                self.trendAnalysis = analysis
                self.isLoading = false
            }
        } catch {
            print("Error fetching historical data: \(error)")
            await MainActor.run {
                self.isLoading = false
            }
        }
    }
    
    func calculateTrendAnalysis(from predictions: [PredictionHistoryEntry]) -> TrendAnalysis {
        guard !predictions.isEmpty else {
            return TrendAnalysis(
                averageLifeExpectancy: 0,
                trendDirection: .stable,
                bestDay: nil,
                worstDay: nil,
                totalChange: 0,
                contributingFactors: []
            )
        }
        
        let sortedPredictions = predictions.sorted { $0.timestamp < $1.timestamp }
        let lifeExpectancies = sortedPredictions.map { $0.lifeExpectancy }
        
        // Calculate average
        let average = lifeExpectancies.reduce(0, +) / Double(lifeExpectancies.count)
        
        // Find best and worst days
        let bestDay = sortedPredictions.max { $0.lifeExpectancy < $1.lifeExpectancy }
        let worstDay = sortedPredictions.min { $0.lifeExpectancy < $1.lifeExpectancy }
        
        // Calculate trend direction
        let firstHalf = Array(lifeExpectancies.prefix(lifeExpectancies.count / 2))
        let secondHalf = Array(lifeExpectancies.suffix(lifeExpectancies.count / 2))
        
        let firstAvg = firstHalf.isEmpty ? 0 : firstHalf.reduce(0, +) / Double(firstHalf.count)
        let secondAvg = secondHalf.isEmpty ? 0 : secondHalf.reduce(0, +) / Double(secondHalf.count)
        
        let trendDirection: TrendDirection
        let changeThreshold = 0.1 // Years
        
        if secondAvg > firstAvg + changeThreshold {
            trendDirection = .improving
        } else if secondAvg < firstAvg - changeThreshold {
            trendDirection = .declining
        } else {
            trendDirection = .stable
        }
        
        // Calculate total change
        let totalChange = lifeExpectancies.last! - lifeExpectancies.first!
        
        // Analyze contributing factors
        let contributingFactors = analyzeContributingFactors(predictions: sortedPredictions)
        
        return TrendAnalysis(
            averageLifeExpectancy: average,
            trendDirection: trendDirection,
            bestDay: bestDay,
            worstDay: worstDay,
            totalChange: totalChange,
            contributingFactors: contributingFactors
        )
    }
    
    private func analyzeContributingFactors(predictions: [PredictionHistoryEntry]) -> [ContributingFactor] {
        var factors: [ContributingFactor] = []
        
        // Analyze health metrics changes
        let healthMetricChanges = calculateHealthMetricChanges(predictions: predictions)
        
        for (metric, change) in healthMetricChanges {
            let impact = calculateImpact(change: change)
            if abs(impact) > 0.05 { // Minimum threshold for significant impact
                factors.append(ContributingFactor(
                    name: metric,
                    impact: impact,
                    description: generateFactorDescription(metric: metric, impact: impact)
                ))
            }
        }
        
        return factors.sorted { abs($0.impact) > abs($1.impact) }.prefix(5).map { $0 }
    }
    
    private func calculateHealthMetricChanges(predictions: [PredictionHistoryEntry]) -> [String: Double] {
        var changes: [String: Double] = [:]
        
        guard predictions.count >= 2 else { return changes }
        
        let first = predictions.first!
        let last = predictions.last!
        
        // Compare key health metrics
        changes["Heart Rate"] = (last.avgHeartRate ?? 0) - (first.avgHeartRate ?? 0)
        changes["Steps"] = (last.dailySteps ?? 0) - (first.dailySteps ?? 0)
        changes["Sleep"] = (last.sleepHours ?? 0) - (first.sleepHours ?? 0)
        changes["Exercise"] = (last.exerciseMinutes ?? 0) - (first.exerciseMinutes ?? 0)
        
        return changes
    }
    
    private func calculateImpact(change: Double) -> Double {
        // Calculate impact based on medical research
        // Changes in life expectancy are typically measured in years
        // Convert to a normalized impact score (0-1 scale)
        let normalizedChange = abs(change) / 10.0 // Assume 10 years is maximum significant change
        return min(normalizedChange, 1.0) * (change > 0 ? 1.0 : -1.0)
    }
    
    private func generateFactorDescription(metric: String, impact: Double) -> String {
        let direction = impact > 0 ? "improved" : "declined"
        let magnitude = abs(impact) > 0.1 ? "significantly" : "slightly"
        
        return "\(metric) has \(magnitude) \(direction), contributing to life expectancy changes"
    }
}

// MARK: - Supporting Models

struct PredictionHistoryEntry: Identifiable, Codable {
    let id = UUID()
    let timestamp: Date
    let lifeExpectancy: Double
    let avgHeartRate: Double?
    let dailySteps: Double?
    let sleepHours: Double?
    let exerciseMinutes: Double?
    let contributingFactors: [String]?
}

struct TrendAnalysis {
    let averageLifeExpectancy: Double
    let trendDirection: TrendDirection
    let bestDay: PredictionHistoryEntry?
    let worstDay: PredictionHistoryEntry?
    let totalChange: Double
    let contributingFactors: [ContributingFactor]
}

struct ContributingFactor {
    let name: String
    let impact: Double
    let description: String
}

enum TrendDirection: String, CaseIterable {
    case improving = "improving"
    case declining = "declining"
    case stable = "stable"
    
    var color: Color {
        switch self {
        case .improving: return .green
        case .declining: return .red
        case .stable: return .orange
        }
    }
    
    var icon: String {
        switch self {
        case .improving: return "arrow.up.right"
        case .declining: return "arrow.down.right"
        case .stable: return "arrow.right"
        }
    }
}

enum HistoricalTimeframe: String, CaseIterable {
    case day = "24h"
    case week = "7d"
    case month = "30d"
    case year = "1y"
    
    var displayName: String {
        switch self {
        case .day: return "Last 24 Hours"
        case .week: return "Last Week"
        case .month: return "Last Month"
        case .year: return "Last Year"
        }
    }
    
    var days: Int {
        switch self {
        case .day: return 1
        case .week: return 7
        case .month: return 30
        case .year: return 365
        }
    }
}